package easik.sketch.util.Export.Constraints;

import java.util.LinkedList;

import easik.sketch.util.Export.ExportConstants;
import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportPath;


/**
 * A class to create a commutative diagram object, including all necessary information gleaned 
 * from the XML file.  Should not be used for any other purpose than database exportation, as it 
 * contains no graphical information.
 * 
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-08-28 Kevin Green
 */
public class ExportCommutativeDiagram extends ExportConstraint {
	
	/**
	 * The name of all procedures created because of a commutative diagram
	 */
	private final static String COMM_PROC_NAME = "commDiag";

	/**
	 * The codomain of the commutative diagram
	 */
	private String targetTable;
	
	/**
	 * Constructor which calls the superclass constructor.
	 * @param id An id number to uniquely identify this constraint from all others.
	 */
	public ExportCommutativeDiagram(int id) {
		super();
		_id = id;
		
	}

	/**
	 * Sets the base table (that is, the domain) of the commutative diagram.
	 */
	public void setBaseTable(){
		ExportPath basePath = (ExportPath)_paths.get(0);
		_baseTable = basePath.getDomain();
	}
	
	/**
	 * Sets the target table (that is, the codomain) of the commutative diagram.
	 */
	public void setTargetTable(){
		ExportPath targetPath = (ExportPath)_paths.get(0);
		targetTable = targetPath.getCoDomain();
	}
	
	/**
	 * Get the target table of the commutative diagram
	 * 
	 * @return The string name of the target table
	 */
	public String getTargetTable(){
		return targetTable;
	}
	
	/**
	 * Sets the constraint string for this constraint
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setConstraintStrings(){
		
		ExportPath pathA = (ExportPath)_paths.get(0);
		ExportPath pathB = (ExportPath)_paths.get(1);
		
		String constraint = _db_name + "_" + COMM_PROC_NAME + _id;
		_a_i_tables.put(_baseTable, constraint);
		
		constraint = ExportConstants.CREATE_PROC + constraint + ExportConstants.PROC_PARAM ;
		constraint += ExportConstants.BEGIN + "DECLARE first, second INT; \n";
		
		constraint += "SELECT " + getSelection(pathA) +"INTO first FROM " + getAllTablesString(pathA) +
							"WHERE " + getAllConditionsString(pathA) + "; "+"\n";
		constraint += "SELECT " +getSelection(pathB) + "INTO second FROM " + getAllTablesString(pathB) +
							"WHERE " + getAllConditionsString(pathB) + "; " +"\n";
		constraint += "IF NOT first <=> second THEN " +
						"call fail('Commutative Diagram Constraint fails.'); \n"+
						"END IF; \n"+ ExportConstants.END ;
		
		_procedureStrings.add(constraint);
	}
	
	/**
	 * Determines the last entity before the codomain of the path and sets the SELECT statement 
	 * to select the reference to the codomain of the path from this entity.
	 * 
	 * @param p The path for which the selection is being made
	 * @return The string of the selection to be made
	 * @since 2006-06-09 Vera Ranieri
	 */
	private String getSelection(ExportPath p){
		String select = new String();
		
		LinkedList edges = p.getEdges();
		ExportEdge e = (ExportEdge)edges.getLast();
		
		String domain = e.getSource();
		String codomain = e.getTarget();
		
		select = domain + "." + codomain + ExportConstants.ID;
		
		return select;
	}
	
	/**
	 * Determines all tables involved in the path that must be referenced by conditions in the WHERE clause.
	 * 
	 * @param p The path for which entities lie
	 * @return The string of where all conditions are taken
	 * @since 2006-06-09 Vera Ranieri
	 */
	private String getAllTablesString(ExportPath p){
		
		//TODO: reformat to use methods available in ExportPath
		LinkedList edges = p.getEdges();
		String allTables = new String();
		
		int size = edges.size();
		
		for(int i = 0; i< size; i++){
			allTables += ((ExportEdge)edges.get(i)).getSource() + ", ";
		}
		
		//Removes last comma and space, then replaces space.
		allTables = allTables.substring(0, allTables.length()-2) + " ";
		return allTables;
	}
	
	/**
	 * Determines all conditions needing to hold along the commutative diagram path.
	 * 
	 * @param p The path for which conditions must hold
	 * @return The string of the conditions
	 * @since 2006-06-09 Vera Ranieri
	 */
	private String getAllConditionsString(ExportPath p){
		String conditions;
		
		LinkedList edges = (LinkedList)p.getEdges();
		int size = edges.size();
		
		//set the first condition (must be present)
		conditions = ((ExportEdge)edges.getFirst()).getSource() + "." 
							+((ExportEdge)edges.getFirst()).getSource() + "_id=id ";
		
		for(int i = 1; i< size; i++){
			String source = ((ExportEdge)edges.get(i-1)).getSource();
			String target = ((ExportEdge)edges.get(i)).getSource(); 
			conditions += "AND " + source + "." + target+"_id=" +
							target + "." +target+"_id ";
		}
		
		return conditions;
	}
}
